//*************************************************************************************************
//
//  Description:
//    LiveryPainter.fx - set of shaders to be used for run-time creation of vinyl-based liveries.
//
//*************************************************************************************************

#include "stddefs.fxh"


//-----------------------------------------------------------------------
//
// Helpers missing in stddefs.fxh.
//

//
// Compile a vertex shader / compile a pixel shader for current platform.
//
// To be used as part of technique pass declaration:
//   VertexShader = COMPILE_VS( MyVertexShader() );
//
#ifdef _PS3_
#define COMPILE_VS( name ) compile sce_vp_rsx name
#define COMPILE_PS( name ) compile sce_fp_rsx name
#else
#define COMPILE_VS( name ) compile vs_3_0 name
#define COMPILE_PS( name ) compile ps_3_0 name
#endif


//
// Low-precision types.
//
// LPFLOAT from sddefs.fxh is no longer available.
//
#ifdef _PS3_
#define Half half
#define Half2 half2
#define Half3 half3
#define Half4 half4
#define Hv( constant ) constant ## h
#else
#define Half float
#define Half2 float2
#define Half3 float3
#define Half4 float4
#define Hv( constant ) constant ## f
#endif


//-----------------------------------------------------------------------
//
// Input parameters.
//


//
// Object space into world space (equal to car space for car at origin) matrix.
// Used: StickerRender, ShapeRender
//
float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


//
// Colour of the part we are filling.
// Used: BackgroundFill, BackgroundFillProjection
//
float3 partColour
<
	string UIName = "Part Colour";
	int coherenceType = ECT_DYNAMIC;
	bool appEdit = true;
> = { 0.0f, 0.0f, 0.0f };


//
// Texture-space offset applied during vertex transformation.
// Used: All
//
float2 vertexOffset
<
	string UIName = "Vertex Offset";
	int coherenceType = ECT_DYNAMIC;
	bool appEdit = true;
> = { 0.0f, 0.0f };


//
// The id of projection to restrict rendering into.
// Used: BackgroundFillProjection, StickerRender, ShapeRender
//
float projectionId
<
	string UIName = "Projection Id";
	bool appEdit = true;
> = 0.0f;


//
// World space into vinyl space matrix.
// Used: StickerRender, ShapeRender
//
float4x4 vinylProjection
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
>;


//
// Colour of the shape we are applying.
// Used: ShapeRender
//
float3 shapeColour
<
	string UIName = "Shape Colour";
	bool appEdit = true;
> = { 0.0f, 0.0f, 0.0f };


//
// Opacity of the shape we are applying.
// Used: ShapeRender
//
float shapeOpacity
<
	string UIName = "Shape Opacity";
	bool appEdit = true;
> = 1.0f;


//-----------------------------------------------------------------------
//
// Input textures.
//

//
// Projection mask texture.
// Encodes classification of pixels in the vinyl map/livery texture into projections (green channel).
// Used: BackgroundFillProjection, StickerRender, ShapeRender
//
texture maskTexture : TEXTURE
<
	string UIName = "Projection Mask Texture";
	bool appEdit = true;
>;


//
// Vinyl texture.
// For stickers, just a colour + opacity texture.
// For shapes, a vector texture with distance in green channel and opacity in alpha.
// Used: StickerRender, ShapeRender
//
texture vinylTexture : TEXTURE
<
	string UIName = "Vinyl Texture";
	bool appEdit = true;
>;


//-----------------------------------------------------------------------
//
// Texture samplers.
//

//
// Sampler for maskTexture.
//
sampler2D maskMap : SAMPLER
<
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "maskTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < maskTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Clamp;
	AddressV = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//
// Sampler for vinylTexture.
//
sampler2D vinylMap : SAMPLER
<
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "vinylTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < vinylTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU = Clamp;
	AddressV = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Vertex shader inputs.
//
// All are the same - they get object space position (to be transformed into car space,
// and used for projection), and some set of UV coordinates (to be used for lookup and rasterization).
//


//
// Get UVs from TEXCOORD2 - for use on bodywork.fx or carbon.fx
//
struct VSINPUT2
{
	float3 position	: POSITION;				// Object-space position.
	float2 texcoord	: TEXCOORD2;			// UVs for vinyl map/livery texture.
};


//
// Get UVs from TEXCOORD3 - for use on glass.fx.
//
struct VSINPUT3
{
	float3 position	: POSITION;				// Object-space position.
	float2 texcoord	: TEXCOORD3;			// UVs for vinyl map/livery texture.
};

//
// Get UVs from TEXCOORD0 - extra; used by CopyTexture.
//
struct VSINPUT0
{
	float3 position	: POSITION;				// View-space position.
	float2 texcoord	: TEXCOORD0;			// UVs for texture access.
};


//-----------------------------------------------------------------------
//
// Vertex shader outputs.
//

//
// Fully-equipped vertex shader output.
// Used: StickerRender, ShapeRender
//
struct VSOUTPUT_VINYL
{
	float4 position	: POSITION;				// View-coordinates position.
	float2 maskUV		: TEXCOORD0;			// UVs in the projection mask.
	float2 vinylUV	: TEXCOORD1;			// Shape/sticker UVs.
};


//
// Only the parameters required for filling a part restricted to a projection.
// Used: BackgroundFillProjection
//
struct VSOUTPUT_FILL_PROJECTION
{
	float4 position	: POSITION;				// View-coordinates position.
	float2 maskUV		: TEXCOORD0;			// UVs in the projection mask.
};


//
// Only the parameters required for filling a part.
// Used: BackgroundFill
//
struct VSOUTPUT_FILL
{
	float4 position	: POSITION;				// View-coordinates position.
};


//
// Only the parameters required for texture processing.
// Used: CopyTexture
//
struct VSOUTPUT_COPY
{
	float4 position		: POSITION;			// View-coordinates position.
	float2 textureUV	: TEXCOORD0;		// UVs in the projection mask.
};


//-----------------------------------------------------------------------
//
// Vertex shader functions.
//
// Building blocks to create vertex shader code from.
// They all use unified interface, to overcome the differences between inputs.
//

//
// "Compute" view-space position - get the UVs in vinyl map/livery texture and interpret like position.
//
float4 VSGetPosition( float3 position, float2 texcoord )
{
	// Apply the offset in UV space.
	float2 offset = texcoord.xy + vertexOffset;
	// Scale and bias (and flip) to convert from UV space [0,1]^2 to view-space [-1,1]^2.
	float x = ( offset.x * 2.0f ) - 1.0f;
	float y = 1.0f - ( offset.y * 2.0f );
	return float4( x, y, 0.0f, 1.0f );
}


//
// Get the UV coordinates to be used to access the projection mask.
//
float2 VSGetMaskUV( float3 position, float2 texcoord )
{
	// Projection mask uses the same addressing as vinyl map.
	return texcoord;
}


//
// Get the UV coordinates of the projected shape/sticker.
//
float2 VSGetVinylUV( float3 position, float2 texcoord )
{
	// Transform from object space into world space.
	float3 positionWS = mul( float4( position, 1.0f ), world ).xyz;
	// Transform from world space into vinyl space.
	float3 result = mul( float4( positionWS, 1.0f ), vinylProjection ).xyz;
	return result.xy;
}


//-----------------------------------------------------------------------
//
// Vertex shaders.
//


//
// Vertex shader for background filling.
// Fill a bodywork.fx-based or carbon.fx-based vehicle part with a solid colour.
// Used: BackgroundFill
//
VSOUTPUT_FILL VSBackgroundFill2( VSINPUT2 input )
{
	VSOUTPUT_FILL output;
	output.position = VSGetPosition( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for background filling.
// Fill a glass.fx-based vehicle part with a solid colour.
// Used: BackgroundFill
//
VSOUTPUT_FILL VSBackgroundFill3( VSINPUT3 input )
{
	VSOUTPUT_FILL output;
	output.position = VSGetPosition( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for background filling with projection restriction.
// Fill a bodywork.fx-based or carbon.fx-based vehicle part with a solid colour.
// Used: BackgroundFillProjection
//
VSOUTPUT_FILL_PROJECTION VSBackgroundFillProjection2( VSINPUT2 input )
{
	VSOUTPUT_FILL_PROJECTION output;
	output.position = VSGetPosition( input.position, input.texcoord );
	output.maskUV = VSGetMaskUV( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for background filling with projection restriction.
// Fill a glass.fx-based vehicle part with a solid colour.
// Used: BackgroundFill, BackgroundFillProjection
//
VSOUTPUT_FILL_PROJECTION VSBackgroundFillProjection3( VSINPUT3 input )
{
	VSOUTPUT_FILL_PROJECTION output;
	output.position = VSGetPosition( input.position, input.texcoord );
	output.maskUV = VSGetMaskUV( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for vinyl rendering.
// Apply a vinyl onto a bodywork.fx-based or carbon.fx-based vehicle part.
// Used: StickerRender, ShapeRender
//
VSOUTPUT_VINYL VSVinylRender2( VSINPUT2 input )
{
	VSOUTPUT_VINYL output;
	output.position = VSGetPosition( input.position, input.texcoord );
	output.maskUV = VSGetMaskUV( input.position, input.texcoord );
	output.vinylUV = VSGetVinylUV( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for vinyl rendering.
// Apply a vinyl onto a glass.fx-based vehicle part.
// Used: StickerRender, ShapeRender
//
VSOUTPUT_VINYL VSVinylRender3( VSINPUT3 input )
{
	VSOUTPUT_VINYL output;
	output.position = VSGetPosition( input.position, input.texcoord );
	output.maskUV = VSGetMaskUV( input.position, input.texcoord );
	output.vinylUV = VSGetVinylUV( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for texture replication.
// Replicate texture onto a bodywork.fx-based or carbon.fx-based vehicle part.
// Used: ReplicateTexture
//
VSOUTPUT_COPY VSReplicateTexture2( VSINPUT2 input )
{
	VSOUTPUT_COPY output;
	output.position = VSGetPosition( input.position, input.texcoord );
	output.textureUV = VSGetMaskUV( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for texture replication.
// Replicate texture onto a glass.fx-based vehicle part.
// Used: ReplicateTexture
//
VSOUTPUT_COPY VSReplicateTexture3( VSINPUT3 input )
{
	VSOUTPUT_COPY output;
	output.position = VSGetPosition( input.position, input.texcoord );
	output.textureUV = VSGetMaskUV( input.position, input.texcoord );
	return output;
}


//
// Vertex shader for texture copying.
// Used: CopyTexture
//
VSOUTPUT_COPY VSCopyTexture( VSINPUT0 input )
{
	VSOUTPUT_COPY output;
	// Apply the offset in UV space.
	float2 offset = input.position.xy + vertexOffset;
	output.position = float4( offset.x, offset.y, 0.0f, 1.0f );
	output.textureUV = input.texcoord;
	return output;
}


//-----------------------------------------------------------------------
//
// Fragment shader output.
//

struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE colour	: COLOR0;
};


//-----------------------------------------------------------------------
//
// Fragment shader functions.
//

//
// Discard the fragment in case it would sample outside of the vinyl.
//
// If you will call this function early in fragment shader, you may save some texture bandwidth,
// as subsequent texture loads on discarded fragments will not generate memory access.
//
// Used: StickerRender, ShapeRender
//
void PSDiscardOutOfVinyl( float2 vinylUV )
{
#ifdef _PS3_
	if ( ( vinylUV.x < 0.0f ) || ( vinylUV.y < 0.0f ) || ( vinylUV.x > 1.0f ) || ( vinylUV.y > 1.0f ) )
	{
		discard;
	}
#else
	float4 error = float4( vinylUV.x, vinylUV.y, ( 1.0f - vinylUV.x ), ( 1.0f - vinylUV.y ) );
	clip( error );
#endif
}


//
// Discard the fragment in case it resides outside of the current projection.
//
// If you will call this function early in fragment shader, you may save some texture bandwidth,
// as subsequent texture loads on discarded fragments will not generate memory access.
//
// Used: BackgroundFillProjection, StickerRender, ShapeRender
//
void PSDiscardOutOfProjection( float2 maskUV )
{
	// We suppot 16 projections at most - currently, there are 11 (+ 1 extra as "none") in use.
	Half PROJECTION_COUNT = Hv(16.0);

	Half4 maskTexel = tex2D( maskMap, maskUV );
	Half mask = maskTexel.g * PROJECTION_COUNT;

	// This is a funny way of saying
	// if ( ( mask < ( projectionId - 0.5 )) || ( mask > projectionId + 0.5 ) ) { discard; }
	Half delta = abs( mask - (Half)( projectionId ) );
	Half error = (Half)( - delta + 0.5f );
#ifdef _PS3_
	if ( error < Hv(0.0) )
	{
		discard;
	}
#else
	clip( error );
#endif

}


//
// Get the partColour value.
// Used: BackgroundFill, BackgroundFillProjection
//
COLOUR_OUTPUT_TYPE PSGetPartColour()
{
	COLOUR_OUTPUT_TYPE result;
	result.rgb = partColour;
	// We are drawing background, thus force 0 for vinyl opacity.
	result.a = Hv(0.0);
	return result;
}


//
// Get the colour of the current sticker.
// Used: StickerRender
//
COLOUR_OUTPUT_TYPE PSGetStickerColour( float2 vinylUV )
{
	COLOUR_OUTPUT_TYPE result;
	result = tex2D( vinylMap, vinylUV );
	// Our colours are too hot and we have to dampen them.
	// Ideally, the source assets would be correct, but instead,
	// we have to scale the colour intensity down in fragment program.
	result.rgb *= Hv(0.55);
	return result;
}


//
// Get the colour of the current shape - single sample.
// Used: ShapeRender
//
COLOUR_OUTPUT_TYPE PSGetShapeColourSimple( float2 vinylUV )
{
	COLOUR_OUTPUT_TYPE result;
	COLOUR_OUTPUT_TYPE texel = tex2D( vinylMap, vinylUV );
	result.rgb = shapeColour;
	Half inVinyl = ( texel.g >= Hv(0.5) ) ? Hv(1.0) : Hv(0.0);
	result.a = shapeOpacity * inVinyl * texel.a;
	return result;
}


//
// Get the colour of the current shape - multiple samples.
// Used: ShapeRender
//
COLOUR_OUTPUT_TYPE PSGetShapeColourMulti( float2 vinylUV )
{
	COLOUR_OUTPUT_TYPE result;
	float2 uvdx = ddx( vinylUV );
	float2 uvdy = ddy( vinylUV );
	float2 sample1 = vinylUV;
	float2 sample2 = vinylUV + uvdx;
	float2 sample3 = vinylUV + uvdy;
	float2 sample4 = vinylUV + uvdx + uvdy;
	COLOUR_OUTPUT_TYPE texel1 = tex2D( vinylMap, sample1 );
	COLOUR_OUTPUT_TYPE texel2 = tex2D( vinylMap, sample2 );
	COLOUR_OUTPUT_TYPE texel3 = tex2D( vinylMap, sample3 );
	COLOUR_OUTPUT_TYPE texel4 = tex2D( vinylMap, sample4 );
	result.rgb = shapeColour;
	Half inVinyl1 = ( texel1.g >= Hv(0.5) ) ? Hv(0.25) : Hv(0.0);
	Half inVinyl2 = ( texel2.g >= Hv(0.5) ) ? Hv(0.25) : Hv(0.0);
	Half inVinyl3 = ( texel3.g >= Hv(0.5) ) ? Hv(0.25) : Hv(0.0);
	Half inVinyl4 = ( texel4.g >= Hv(0.5) ) ? Hv(0.25) : Hv(0.0);
	Half inVinyl = inVinyl1 + inVinyl2 + inVinyl3 + inVinyl4;
	result.a = shapeOpacity * inVinyl * texel1.a;
	return result;
}


//
// Get the colour of the current shape - fixed width outline.
// Used: ShapeRender
//
COLOUR_OUTPUT_TYPE PSGetShapeColourFixedWidth( float2 vinylUV )
{
	COLOUR_OUTPUT_TYPE result;
	COLOUR_OUTPUT_TYPE texel = tex2D( vinylMap, vinylUV );
	result.rgb = shapeColour;
	Half inVinyl = saturate( ( texel.g - Hv(0.47) ) / ( Hv(0.53) - Hv(0.47) ) );
	result.a = shapeOpacity * inVinyl * texel.a;
	return result;
}


//
// Get the colour of the current shape - variable width outline.
// Used: ShapeRender
//
COLOUR_OUTPUT_TYPE PSGetShapeColourVariableWidth( float2 vinylUV )
{
	COLOUR_OUTPUT_TYPE result;
	COLOUR_OUTPUT_TYPE texel = tex2D( vinylMap, vinylUV );
	result.rgb = shapeColour;
	float2 derivatives = max( abs( ddx( vinylUV ) ), abs( ddy( vinylUV ) ) );
	float width = saturate( max( derivatives.x, derivatives.y ) );
	float delta = 0.4f * width * 4.0f;
	Half inVinyl = saturate( ( texel.g - ( Hv(0.5) - delta ) ) / ( Hv(2.0) * delta ) );
	result.a = shapeOpacity * inVinyl * texel.a;
	return result;
}


//
// Get the colour of the current shape - constant width outline.
// Used: ShapeRender
//
COLOUR_OUTPUT_TYPE PSGetShapeColourConstantWidth( float2 vinylUV )
{
	COLOUR_OUTPUT_TYPE result;
	COLOUR_OUTPUT_TYPE texel = tex2D( vinylMap, vinylUV );
	result.rgb = shapeColour;

	// FIXME: So far hardcoded.
	float scale = 128.0f;
	float2 texelUV = vinylUV * float2( scale, scale );
	float4 texelDerivatives = float4( ddx( texelUV ), ddy( texelUV ) );

	// Note: We are not taking mipmaps into consideration...

	float trace = dot( texelDerivatives, texelDerivatives );

	float diagonal1 = texelDerivatives.x * texelDerivatives.w;
	float diagonal2 = texelDerivatives.y * texelDerivatives.z;
	float diagonalBoth = diagonal1 * diagonal2;
	float determinant = diagonal1 * diagonal1 + diagonal2 * diagonal2 - 2.0f * diagonalBoth;
	float bias = trace * trace - 4.0f * determinant;

	float eigenValue1 = 0.5f * ( trace + sqrt( bias ) );
	float eigenValue2 = 0.5f * ( trace - sqrt( bias ) );

	float diameter1 = 1.0f / sqrt( eigenValue1 );
	float diameter2 = 1.0f / sqrt( eigenValue2 );

	// Use the major diameter to favour sharpness.

	float texelSize = max( diameter1, diameter2 );
	float width = 1.0f / 10.0f;				// Tuned to look good.
	float contrast = max( 1.0f, texelSize / width );

	Half inVinyl = saturate( 0.5f + contrast * ( texel.g - 0.5f ) );
	result.a = shapeOpacity * inVinyl * texel.a;
	return result;
}


#define PSGetShapeColour PSGetShapeColourConstantWidth


//-----------------------------------------------------------------------
//
// Fragment shaders.
//

//
// Pixel shader for background filling.
// Fill the area to solid colour from partColour.
// Used: BackgroundFill
//
PSOUTPUT PSBackgroundFill( VSOUTPUT_FILL input )
{
	PSOUTPUT output;
	output.colour = PSGetPartColour();
	return output;
}


//
// Pixel shader for background filling restricted to a projection.
// Fill the area to solid colour from partColour.
// Used: BackgroundFillProjection
//
PSOUTPUT PSBackgroundFillProjection( VSOUTPUT_FILL_PROJECTION input )
{
	PSOUTPUT output;
	PSDiscardOutOfProjection( input.maskUV );
	output.colour = PSGetPartColour();
	return output;
}


//
// Pixel shader for sticker rendering.
// Just apply the sticker texture directly, no modifications.
// Used: StickerRender
//
PSOUTPUT PSStickerRender( VSOUTPUT_VINYL input )
{
	PSOUTPUT output;
	PSDiscardOutOfVinyl( input.vinylUV );
	PSDiscardOutOfProjection( input.maskUV );
	output.colour = PSGetStickerColour( input.vinylUV );
	return output;
}


//
// Pixel shader for shape rendering.
// Apply shape colour and shape opacity.
// Used: ShapeRender
//
PSOUTPUT PSShapeRender( VSOUTPUT_VINYL input )
{
	PSOUTPUT output;
	PSDiscardOutOfVinyl( input.vinylUV );
	PSDiscardOutOfProjection( input.maskUV );
	output.colour = PSGetShapeColour( input.vinylUV );
	return output;
}


//
// Pixel shader for texture copying.
// Used: CopyTexture
//
PSOUTPUT PSCopyTexture( VSOUTPUT_COPY input )
{
	PSOUTPUT output;
	output.colour = tex2D( vinylMap, input.textureUV );
	return output;
}


//-----------------------------------------------------------------------
//
// Techniques.
//
// There is a lot of boilerprate, despite all the techniques being the same.
// So let us resort to use of preprocessor macros.
//

//
// Declare a technique with standard minimal set of annotations.
// Note that we do not preserve global state.
//
#define TECHNIQUE( name ) \
	technique name \
	< \
		bool supportsSpecialisedLighting = false; \
	  bool preservesGlobalState = false; \
		string normalBehaviour = "ERMB_RENDER"; \
		string normalTechnique = #name; \
		int    normalDeferredID = 0; \
		bool isBillboard = true; \
	>


//
// Declare rendering states for overwriting framebuffer.
// Used: BackgroundFill, BackgroundFillProjection
//
#ifdef _PS3_
	#define TECHNIQUE_STATE_OVERWRITE \
		ZEnable = false; \
		ZWriteEnable = false; \
		AlphaBlendEnable = false; \
		CullFaceEnable = false;
#else		// ! _PS3_
	#define TECHNIQUE_STATE_OVERWRITE \
		ZEnable = false; \
		ZWriteEnable = false; \
		AlphaBlendEnable = false; \
		CullMode = None;
#endif		// ! _PS3_


//
// Declare rendering states for blending colour into framebuffer and summing alpha.
// Used: none
//
#ifdef _PS3_
	#define TECHNIQUE_STATE_BLEND_PREMULTIPLIED \
		ZEnable = false; \
		ZWriteEnable = false; \
		AlphaBlendEnable = true; \
		CullFaceEnable = false; \
		BlendFuncSeparate = int4( One, OneMinusSrcAlpha, One, One ); \
		BlendEquationSeparate = int2( FuncAdd, FuncAdd );
#else		// ! _PS3_
	#define TECHNIQUE_STATE_BLEND_PREMULTIPLIED \
		ZEnable = false; \
		ZWriteEnable = false; \
		CullMode = None; \
		AlphaBlendEnable = true; \
		DestBlend = INVSRCALPHA; \
		SrcBlend = ONE; \
		BlendOp = ADD; \
		SeparateAlphaBlendEnable = true; \
		DestBlendAlpha = ONE; \
		SrcBlendAlpha = ONE; \
		BlendOpAlpha = ADD;
#endif		// ! _PS3_


//
// Declare rendering states for blending colour into framebuffer and summing alpha.
// Used: ShapeRender, StickerRender
//
#ifdef _PS3_
	#define TECHNIQUE_STATE_BLEND \
		ZEnable = false; \
		ZWriteEnable = false; \
		AlphaBlendEnable = true; \
		CullFaceEnable = false; \
		BlendFuncSeparate = int4( SrcAlpha, OneMinusSrcAlpha, One, One ); \
		BlendEquationSeparate = int2( FuncAdd, FuncAdd );
#else		// ! _PS3_
	#define TECHNIQUE_STATE_BLEND \
		ZEnable = false; \
		ZWriteEnable = false; \
		CullMode = None; \
		AlphaBlendEnable = true; \
		DestBlend = INVSRCALPHA; \
		SrcBlend = SRCALPHA; \
		BlendOp = ADD; \
		SeparateAlphaBlendEnable = true; \
		DestBlendAlpha = ONE; \
		SrcBlendAlpha = ONE; \
		BlendOpAlpha = ADD;
#endif		// ! _PS3_


//
// All the variants of BackgroundFill.
//

TECHNIQUE( BackgroundFill2 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_OVERWRITE
		VertexShader = COMPILE_VS( VSBackgroundFill2() );
		PixelShader = COMPILE_PS( PSBackgroundFill() );
	}
}

TECHNIQUE( BackgroundFill3 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_OVERWRITE
		VertexShader = COMPILE_VS( VSBackgroundFill3() );
		PixelShader = COMPILE_PS( PSBackgroundFill() );
	}
}


//
// All the variants of BackgroundFillProjection.
//

TECHNIQUE( BackgroundFillProjection2 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_OVERWRITE
		VertexShader = COMPILE_VS( VSBackgroundFillProjection2() );
		PixelShader = COMPILE_PS( PSBackgroundFillProjection() );
	}
}

TECHNIQUE( BackgroundFillProjection3 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_OVERWRITE
		VertexShader = COMPILE_VS( VSBackgroundFillProjection3() );
		PixelShader = COMPILE_PS( PSBackgroundFillProjection() );
	}
}


//
// All the variants of StickerRender.
//

TECHNIQUE( StickerRender2 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_BLEND
		VertexShader = COMPILE_VS( VSVinylRender2() );
		PixelShader = COMPILE_PS( PSStickerRender() );
	}
}

TECHNIQUE( StickerRender3 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_BLEND
		VertexShader = COMPILE_VS( VSVinylRender3() );
		PixelShader = COMPILE_PS( PSStickerRender() );
	}
}


//
// All the variants of ShapeRender.
//

TECHNIQUE( ShapeRender2 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_BLEND
		VertexShader = COMPILE_VS( VSVinylRender2() );
		PixelShader = COMPILE_PS( PSShapeRender() );
	}
}

TECHNIQUE( ShapeRender3 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_BLEND
		VertexShader = COMPILE_VS( VSVinylRender3() );
		PixelShader = COMPILE_PS( PSShapeRender() );
	}
}


//
// All the variants of ReplicateTexture.
//

TECHNIQUE( ReplicateTexture2 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_OVERWRITE
		VertexShader = COMPILE_VS( VSReplicateTexture2() );
		PixelShader = COMPILE_PS( PSCopyTexture() );
	}
}

TECHNIQUE( ReplicateTexture3 )
{
	pass Pass0
	{
		TECHNIQUE_STATE_OVERWRITE
		VertexShader = COMPILE_VS( VSReplicateTexture3() );
		PixelShader = COMPILE_PS( PSCopyTexture() );
	}
}


//
// Extra techniques for texture processing.
//
TECHNIQUE( CopyTexture )
{
	pass Pass0
	{
		TECHNIQUE_STATE_OVERWRITE
		VertexShader = COMPILE_VS( VSCopyTexture() );
		PixelShader = COMPILE_PS( PSCopyTexture() );
	}
}


// eof
